//----------------------------------------------------------//
//--------- Chat message handle, parse and decorate --------//
//----------------------------------------------------------//
const MSG_PREFIX_MEDIA_SOUND = 'sound=';
const MSG_PREFIX_MEDIA_YT = 'yt=';
const MSG_PREFIX_IMG_CIRCLE = 'img-circle=';
const MSG_PREFIX_IMG_BG = 'img-wide=';
const MSG_PREFIX_IMG_REMOVABLE = 'img=';

// whether or not Roll20 Chat content is loaded
var isChatInit = document.querySelector('.userscript-commandintro') !== null;
				
//------------------------------------------------------------
// Create an audio container into the chat with auto-play if needed
//------------------------------------------------------------
function createAudio(node, src, hasToPlay) {
	// create audio HTML element
	var audio = createCustomElement("audio");
	audio.controls = 'controls';
	audio.autoplay = hasToPlay;
	audio.classList.add('audio_msg');
	audio.style.width='100%';
	audio.load();
	audio.addEventListener("error", function() {
		node.parentElement.removeChild(audio);
	}, true);
	audio.onended = function() {
		node.parentElement.removeChild(audio);
	};
	node.parentElement.appendChild(audio);
	var elemChat = document.getElementById('textchat');
	elemChat.scrollTo(0,elemChat.scrollHeight);
	// add source
	audio.innerHTML = '<source src="' + src + '">';
}

//------------------------------------------------------------
// parse chat node case : 
// Display an audio container into the chat with auto-play if needed
//------------------------------------------------------------
function parseAndPlaySound(node, hasToPlay) {

	var message = node.innerHTML;
	var idx = message.indexOf(MSG_PREFIX_MEDIA_SOUND);
	if(idx >= 0) {
		if (!isChatInit) {
			// loading page => remove node in chat
			node.parentNode.removeChild(node);
			return true;
		} else {
			let src = message.substring(idx + 6).trim();
			if (src.indexOf('http') != 0) {
				src = 'http://sda-campagne.fr/misc/' + src;
			}
			node.innerHTML = node.innerHTML.substring(0, idx);
			var soundIcon = createCustomElement('span');
			soundIcon.classList.add('pictos');
			soundIcon.style = 'font-size:18px; cursor: pointer!important; color:#742d2d';
			soundIcon.title = src;
			soundIcon.onclick = function() {
				createAudio(node, src, true);
			};
			soundIcon.innerHTML = '&gt;';
			node.appendChild(soundIcon);
			// parse URL
			createAudio(node, src, hasToPlay);
		}
		return true;
	}
}

//------------------------------------------------------------
// parse chat node case : 
// display an image as circle in middle of the screen
//------------------------------------------------------------
function parseAndDisplayCircleImage(node) {
	// display image in circle (greek theme only)

	var message = node.innerHTML;
	var idx = message.indexOf(MSG_PREFIX_IMG_CIRCLE);
	if(idx >= 0) {
		if (!isChatInit) {
			// loading page => remove node in chat
			node.parentNode.removeChild(node);
		} else {
			console.log('---------------- img-circle');
			// parse background-image
			var url = message.substring(idx + 11).trim();
			console.log('---------------- img-circle URL : ' + url);
			console.log('---------------- img-circle HTML:' + node.innerHTML);
			console.log('---------------- img-circle : ' + node);
			node.parentNode.removeChild(node);

			var circleToggleBtn = document.getElementById('circle-toggle-btn');
			var circleContainer = document.getElementById('circle-container');
			switch(url) {
				case 'on':
					// set circle visibility as ON
					circleContainer.style.opacity = 1;
					circleToggleBtn.innerHTML = 'Hide Circle';
					circleToggleBtn.style.boxShadow = '0 0 25px black inset, 0 0 25px black inset';
					break;
				case 'off':
					// set circle visibility as OFF
					circleContainer.style.opacity = 0;
					circleToggleBtn.innerHTML = 'Show Circle';
					circleToggleBtn.style.boxShadow = '0 0 25px white inset, 0 0 25px white inset';
					break;
				default:
					// change image in circle
					var circleView = document.getElementById('circle-view');
					circleView.style.backgroundImage = 'url("' + url + '")';
					circleToggleBtn.style.backgroundImage = 'url("' + url + '")';
					break;
			}
		}
		return true;
	}
	return false;
}

//------------------------------------------------------------
// parse chat node case : 
// display an image as background
//------------------------------------------------------------
function parseAndDisplayBackgroundImage(node) {

	var message = node.innerHTML;
	var idx = message.indexOf(MSG_PREFIX_IMG_BG);
	if(idx >= 0) {
		if (!isChatInit) {
			// loading page => remove node in chat
			node.parentNode.removeChild(node);
		} else {
			console.log('---------------- img-wide');
			// parse background-image
			var url = message.substring(idx + 9).trim();
			console.log('---------------- img-wide URL : ' + url);
			console.log('---------------- img-wide HTML:' + node.innerHTML);
			console.log('---------------- img-wide : ' + node);
			node.parentNode.removeChild(node);

			// change image in background
			var wideView = document.getElementById('wide-view');
			var wideHideBtn = document.getElementById('wide-hide-btn');
			switch(url) {
				case 'off':
					// set visibility as OFF
					wideView.style.opacity = 0;
					wideHideBtn.style.backgroundImage = 'none';
					break;
				default:
					// change image
					wideHideBtn.style.backgroundImage = 'url("' + url + '")';
					wideView.style.opacity = '0';
					setTimeout(() => {
						wideView.src = url;
						wideView.style.opacity = '1';
					}, 300);
					break;
			}
		}
		return true;
	}
	return false;
}

//------------------------------------------------------------
// Display a wide image removable by user click
//------------------------------------------------------------
function displayImg(src) {
	// hide Roll20 editor wrapper
	var wrapper = document.getElementById('editor-wrapper');
	var pageToolbar = document.getElementById('page-toolbar');
	var zoomslider = document.getElementById('zoomslider');
	wrapper.classList.add('displaying-img');
	// create wide image
	var img = createCustomElement('IMG');
	img.classList.add('closable-img');
	img.src = src;
	img.style.transition = 'all 0.5s ease-in-out';
	img.style.cursor = 'pointer';
	img.style.opacity = '0';
	img.onclick = function() {
		img.style.opacity = '0';
		setTimeout(function(){
			wrapper.removeChild(img);
			wrapper.classList.remove('displaying-img');
		}, 500);
	}
	// remove previous displayed image
	for(var child of wrapper.children) {
		if(child.tagName && child.tagName == 'IMG'){
			wrapper.removeChild(child);
		}
	}
	// display new image
	wrapper.appendChild(img);
	setTimeout(function(){
		img.style.opacity = '1';
	}, 100);
}

//------------------------------------------------------------
// parse chat node case : 
// Image display - removable by user click
// 1- create a thumbnail in chat to display image
// 2- if hasToDisplay true => display wide image
//------------------------------------------------------------
function parseAndDisplayRemovableImage(node, hasToDisplay) {
	// display image by chat

	var message = node.innerHTML;
	var idx = message.indexOf(MSG_PREFIX_IMG_REMOVABLE);
	if(idx >= 0) {
		if (!isChatInit) {
			// loading page => remove node in chat
			node.parentNode.removeChild(node);
		} else {
			// parse URL
			var url = message.substring(idx + 4).trim();
			if(url.indexOf('http') != 0){
				url = 'http://sda-campagne.fr/misc/' + url;
			}
			node.innerHTML = node.innerHTML.substring(0, idx);

			// build image thumbnail in chat
			var imgIcon = createCustomElement('span');
			imgIcon.classList.add('pictos');
			imgIcon.style = 'font-size:18px; cursor: pointer!important; color:#742d2d';
			imgIcon.title = url;
			imgIcon.onclick = function() {
				displayImg(url);
			};
			imgIcon.innerHTML = 'P';
			node.appendChild(imgIcon);

			// build mini img
			var imgChat = createCustomElement('img');
			imgChat.src = url;
			imgChat.style.cursor = 'pointer';
			imgChat.onclick = function() {
				displayImg(url);
			};
			node.appendChild(imgChat);

			// display image
			if(hasToDisplay) {
				displayImg(url);
			}
		}
		return true;
	}
	return false;
}

//------------------------------------------------------------
// Create YT media to play in background
//------------------------------------------------------------
function createAudioYT(node, src, hasToPlay) {
	var idx = src.indexOf('watch?v=');
	if (idx >= 0) {
		var substr = src.substring(idx + 8);
		if (substr.indexOf('&') > 0) {
			substr = substr.substring(0, substr.indexOf('&'));
		}
		src = src.substring(0, idx) + 'embed/' + substr + '?enablejsapi=1&autoplay=1&color=white&fs=0&modestbranding=1&version=3&showinfo=0';
	} else {
		alert('url synthax error! (missing watch?v=)');
		return;
	}
	node.remove();

	var jukebox = document.getElementById('jukebox');
	var jukeboxContentFirstChild = jukebox.children[0].children[0];
	if (jukeboxContentFirstChild.id != 'ytContainer') {
		// create audio HTML element
		var audioYT = createCustomElement("iframe");
		audioYT.id = "ytPlayer";
		audioYT.style.width = "100%";
		audioYT.style.height = "100%";
		audioYT.style.position= 'absolute';
		audioYT.style.top= '0';
		audioYT.style.left= '0';
		audioYT.style.border= 'none';
		audioYT.src = src;
		audioYT.frameborder = "0";
		audioYT.type = "text/html";

		var container = createCustomElement("div");
		container.id = 'ytContainer';
		container.style = 'position: absolute; width: 400px; height: 226px; left: -462px; top: 0px; border: 5px solid white; border-radius: 25px;overflow: hidden; box-shadow: 0 1px 16px rgba(0, 0, 0, 0.46), 0 15px 15px -12px;';

		container.appendChild(audioYT);

		// add youtube player to jukebox container
		jukebox.children[0].prepend(container);
		jukebox.style.overflow = 'visible';
		document.getElementById('rightsidebar').style.overflow = 'visible';
	} else {
		jukeboxContentFirstChild.children[0].src = src;
	}
}

//------------------------------------------------------------
// parse node case : 
// YT media to play in background
//------------------------------------------------------------
function parseAndPlayYT(node, hasToPlay) {
	var message = node.innerHTML;
	var idx = message.indexOf(MSG_PREFIX_MEDIA_YT);
	if(idx >= 0) {
		let src = message.substring(idx + 3).trim();
		node.innerHTML = node.innerHTML.substring(0, idx);
		var soundIcon = createCustomElement('span');
		soundIcon.classList.add('pictos');
		soundIcon.style = 'font-size:18px; cursor: pointer!important; color:#742d2d';
		soundIcon.title = src;
		soundIcon.innerHTML = '&gt;';
		node.appendChild(soundIcon);
		// parse URL
		createAudioYT(node, src, hasToPlay);
	}
}

//------------------------------------------------------------
// parse node case : dice roll result msg 
// => decorate chat node elt
//------------------------------------------------------------
function parseAndDecorateDiceRollResult(node) {
	console.log('parseAndDecorateDiceRollResult...');
	var message = node.innerHTML;
	var idx = message.indexOf('[dice-roll-bol]:');
	if(idx >= 0) {
	console.log('parseAndDecorateDiceRollResult...msg identified');
		const dataStr = message.substring(idx + 16).trim();
		const data = JSON.parse(dataStr);

	console.log('parseAndDecorateDiceRollResult...data: ' + JSON.stringify(data));
		var rollDetails = createCustomElement('div');
		rollDetails.classList.add('dice-roll-bol');
		rollDetails.classList.add(data.type);
		var innerHTML = '<div class="roll-auth">' + data.auth + '</div>';
		innerHTML += '<div class="roll-type">';
		innerHTML += '<span class="roll-attribute-value">' + (data.attribute.value >= 0 ? ('+' + data.attribute.value) : data.attribute.value) + '</span>';
		innerHTML += data.attribute.name + ' + ' + data.skill.name;
		innerHTML += '<span class="roll-skill-value">' + (data.skill.value >= 0 ? ('+' + data.skill.value) : data.skill.value) + '</span>';
		innerHTML += '</div>';
		innerHTML += '<div class="roll-res">' + data.result + '</div>';
		innerHTML += '<div class="roll-modif">' + data.diff + '</div>';
		innerHTML += '<div class="roll-details">';
		for (i in data.dices) {
			if (data.diceRollLevel == 0) {
				innerHTML += '<span class="dice-res">' + data.dices[i] + '</span>';
			} else if (data.diceSelection[i]) {
				innerHTML += '<span class="'+ (data.diceRollLevel > 0 ? 'dice-res bonus' : 'dice-res malus') + '">' + data.dices[i] + '</span>';
			} else {
				innerHTML += '<span class="dice-res ignored">' + data.dices[i] + '</span>';
			}
		}
		innerHTML += '<span style="padding-left: 10px">' + (data.modif >= 0 ? ('+' + data.modif) : data.modif) + '</span>';
		innerHTML += '</div>';
		rollDetails.innerHTML = innerHTML;

		node.innerHTML = '';
	console.log('parseAndDecorateDiceRollResult...node A: ' + node.innerHTML);
		node.appendChild(rollDetails);
	console.log('parseAndDecorateDiceRollResult...node B: ' + node.innerHTML);
	}
}

//-----------------------------------------------------------------------
// observer on chat content -> parse to play media or display image
//-----------------------------------------------------------------------
var messageObserver = new MutationObserver(function (mutationsList) {
	for(var mutation of mutationsList) {
		if (mutation.type == 'childList') {
			// parse new node added into the chat
			for (var node of mutation.addedNodes) {
				// check init to avoid auto-play media from old msg
				if (node.nodeType === Node.ELEMENT_NODE) {
					if (!isChatInit) {
						isChatInit = document.querySelector('.userscript-commandintro') !== null;

						// play sound
						if(parseAndPlaySound(node, false)) {
							continue;
						}
						// display image
						if(parseAndDisplayRemovableImage(node, false)) {
							continue;
						}
					} else {
						// case : audio sound
						if(parseAndPlaySound(node, true)) {
							continue;
						}
						// case : YT video
						if(parseAndPlayYT(node, true)) {
							continue;
						}
						// case : image removable
						if(parseAndDisplayRemovableImage(node, true)) {
							continue;
						}
					}
					// display dice bol roll result
					if(parseAndDecorateDiceRollResult(node)) {
					   continue;
					}
					// case : image into circle
					if(parseAndDisplayCircleImage(node, true)) {
						continue;
					}
					// case : image in background
					if(parseAndDisplayBackgroundImage(node, true)) {
						continue;
					}
				}
			}
		}
	}
});
waitElementExist('textchat', () => {
	let elemChat = document.getElementById('textchat');
	for (var i = 0; i < elemChat.children.length; i++) {
		var e = elemChat.children[i];
		if (e.classList.contains('content')) {
			messageObserver.observe(e, {
				childList: true
			});
			break;
		}
	}
});
